----------------------------------------------------------------------
--  Adasubst - Main program body                                    --
--  Copyright (C) 2002, 2019 Adalog                                 --
--  Author: J-P. Rosen                                              --
--                                                                  --
--  ADALOG   is   providing   training,   consultancy,   expertise, --
--  assistance and custom developments  in Ada and related software --
--  engineering techniques.  For more info about our services:      --
--  ADALOG                   Tel: +33 1 41 24 31 40                 --
--  19-21 rue du 8 mai 1945  Fax: +33 1 41 24 07 36                 --
--  94110 ARCUEIL            E-m: info@adalog.fr                    --
--  FRANCE                   URL: http://www.adalog.fr              --
--                                                                  --
--  This  unit is  free software;  you can  redistribute  it and/or --
--  modify  it under  terms of  the GNU  General Public  License as --
--  published by the Free Software Foundation; either version 2, or --
--  (at your  option) any later version.  This  unit is distributed --
--  in the hope  that it will be useful,  but WITHOUT ANY WARRANTY; --
--  without even the implied warranty of MERCHANTABILITY or FITNESS --
--  FOR A  PARTICULAR PURPOSE.  See the GNU  General Public License --
--  for more details.   You should have received a  copy of the GNU --
--  General Public License distributed  with this program; see file --
--  COPYING.   If not, write  to the  Free Software  Foundation, 59 --
--  Temple Place - Suite 330, Boston, MA 02111-1307, USA.           --
--                                                                  --
--  As  a special  exception, if  other files  instantiate generics --
--  from  this unit,  or you  link this  unit with  other  files to --
--  produce an executable,  this unit does not by  itself cause the --
--  resulting executable  to be covered  by the GNU  General Public --
--  License.  This exception does  not however invalidate any other --
--  reasons why  the executable  file might be  covered by  the GNU --
--  Public License.                                                 --
----------------------------------------------------------------------

with   -- Standard Ada units
  Ada.Calendar,
  Ada.Characters.Handling,
  Ada.Command_Line,
  Ada.Exceptions;

with   -- ASIS components
  Asis.Errors,
  Asis.Exceptions,
  Asis.Implementation;

with   -- Other reusable components
  Implementation_Options,
  Options_Analyzer,
  Project_File.Factory,
  Units_List,
  Utilities;

with   -- Application specific units
  Dictionary,
  Tools.Dependencies,
  Tools.Instantiator,
  Tools.Preparator,
  Tools.Translator,
  Tools.Unrepresenter,
  Tools.Unuser;
procedure Adasubst is
   use Ada.Calendar;
   use Utilities, Implementation_Options;
   use Asis.Implementation,Asis.Exceptions;

   Version             : constant Wide_String := "1.6r5";
   Default_Line_Length : constant := 200;
   Min_Line_Length     : constant := 80;
   Start_Time          : constant Time := Clock;
   Unit_Count          : Natural := 0;

   package Options is new Options_Analyzer (Binary_Options => "hbcdrsuvw",
                                            Valued_Options => "loOp",
                                            Tail_Separator => "--");
   use Options;

   type Action_Kinds is (Help, Dependents, Instantiate, Prepare, Translate, Unrepresent, Unuse);
   Action               : Action_Kinds;
   Body_Option          : Boolean;
   Spec_Option          : Boolean;
   Comment_Option       : Boolean;
   Recursive_Option     : Boolean;
   Overwrite_Option     : Boolean;
   Unit_Specification   : Positive;
   Dictionary_Name      : Positive;
   Line_Length          : Natural;
   Which_O_Option       : Character := 'o';

   procedure Analyze_Options is
      L_Value : Integer;
   begin
      if Is_Present (Option => 'h') then
         Action := Help;
         return;
      end if;

      if Parameter_Count = 0 then
         raise Options_Error with "Missing command name";
      end if;

      -- Determine Action
      begin
         Action := Action_Kinds'Value (Parameter (1));
      exception
         when Constraint_Error =>
            raise Options_Error with "Incorrect function: " & Parameter (1);
      end;

      -- Initialize options
      Utilities.Debug_Option   := Is_Present (Option => 'd');
      Utilities.Error_Is_Out   := not (Is_Present (Option => 'o') or Is_Present (Option => 'O'));
      Utilities.Verbose_Option := Is_Present (Option => 'v');

      Overwrite_Option := Is_Present (Option => 'w');
      Comment_Option   := Is_Present (Option => 'c');
      Recursive_Option := Is_Present (Option => 'r');


      Spec_Option      := Is_Present (Option => 's');
      Body_Option      := Is_Present (Option => 'b');
      -- If no -s or -b option provided, handle both
      if not (Body_Option or Spec_Option) then
         Spec_Option := True;
         Body_Option := True;
      end if;

      -- No -l option : Length is bounded to the default
      -- -l option without value : Length is unbounded
      if Is_Present (Option => 'l') then
         L_Value := Value (Option => 'l', Default => 0);
         if L_Value < Min_Line_Length and L_Value /= 0 then
            raise Options_Error with "Line length too small:" & Integer'Image (L_Value);
         end if;
         Line_Length := L_Value;
      else
         Line_Length := Default_Line_Length;
      end if;

      Tools.Unuser.Use_Type_Option := Is_Present (Option => 'u');

      if Is_Present (Option => 'O') then
         if Is_Present (Option => 'o') then
            raise Options_Error with "Only one of 'o' and 'O' allowed";
         end if;
         Which_O_Option := 'O';
      end if;

      -- Action dependent checks
      case Action is
         when Help =>
            null;

         when Dependents =>
            if Parameter_Count /= 2 then
               raise Options_Error with "One and only one parameter required";
            end if;

            Unit_Specification := 2;
            Recursive_Option   := True;

         when Instantiate | Prepare | Unrepresent | Unuse =>
            if Parameter_Count /= 2 then
               raise Options_Error with "One and only one parameter required";
            end if;

            Unit_Specification := 2;

         when Translate =>
            if Parameter_Count /= 3 then
               raise Options_Error with "Two and only two parameters required";
            end if;

            Dictionary_Name    := 2;
            Unit_Specification := 3;
      end case;
   end Analyze_Options;

   procedure Print_Help (Version_Only : Boolean := False) is
   begin
      User_Message ("ADASUBST V" & Version & " with " & ASIS_Implementor_Version );
      if Version_Only then
         return;
      end if;

      User_Message ("Copyright (c) 2002-2017 Adalog");
      User_Message ("For information about Adalog's services, please see http://www.adalog.fr");
      User_Message ("Usage:");
      User_Message ("  adasubst translate   [-bcdrsvw]  [-oO <out-prefix>] [-p <project-file>] [-l <line-length>]");
      User_Message ("                       <dictionary> <unit>{+|-<unit>}|@<file> [-- <ASIS options>]");
      User_Message ("  adasubst instantiate [-bcdrsvw]  [-oO <out-prefix>] [-p <project-file>] <unit>{+|-<unit>}" &
                      " [-- <ASIS options>]");
      User_Message ("  adasubst unrepresent [-bcdrsvw]  [-oO <out-prefix>] [-p <project-file>] <unit>{+|-<unit>}" &
                                            " [-- <ASIS options>]");
      User_Message ("  adasubst unuse       [-bcdrsuvw] [-oO <out-prefix>] [-p <project-file>] <unit>{+|-<unit>}" &
                                            " [-- <ASIS options>]");
      User_Message ("  adasubst prepare     [-bdrsvw]   [-o <out-file>] [-p <project-file>] <unit>{+|-<unit>}" &
                                            " [-- <ASIS options>]");
      User_Message ("  adasubst dependents  [-bdrsw]    [-o <out-file>] [-p <project-file>] <unit>{+|-<unit>}" &
                                            " [-- <ASIS options>]");
      User_Message ("  adasubst help");
   end Print_Help;

   Recursion_Modes : constant array (Boolean) of Units_List.Recursion_Mode := (False => Units_List.None,
                                                                               True  => Units_List.Full);
   use Ada.Characters.Handling;
begin  -- Adasubst
   Analyze_Options;

   case Action is
      when Help =>
         Print_Help;
         return;
      when Instantiate =>
         Tools.Instantiator.Prepare;
      when Prepare | Dependents | Unrepresent | Unuse =>
         null;
      when Translate =>
         Dictionary.Initialize (Parameter (Dictionary_Name));
   end case;

   --
   -- Initialize environment
   --

   begin
      Tools.Initialize (Parameters_String
                        (Project => Project_File.Factory.Corresponding_Project (Value (Option => 'p',
                                                                                       Explicit_Required => True)),
                         Other_Options => To_Wide_String (Options.Tail_Value)));
   exception
      when Implementation_Error =>
         -- -p file not found
         User_Message ("Parameter file not found");
         return;
   end;

   Units_List.Register (To_Wide_String (Parameter (Unit_Specification)),
                        Recursion => Recursion_Modes (Recursive_Option),
                        Add_Stubs => True);

   Units_List.Reset;
   while not Units_List.Is_Exhausted loop
      Unit_Count := Unit_Count + 1;
      case Action is
         when Help =>
            -- Impossible, there is a return on top
            raise Program_Error;
         when Prepare =>
            Tools.Preparator.Process(Unit_Name        => Units_List.Current_Unit,
                                     Output_Name      => Value (Option            => Which_O_Option,
                                                                Explicit_Required => True),
                                     Overwrite_Option => Overwrite_Option);

         when Dependents =>
            Tools.Dependencies.Process (Unit_Name        => Units_List.Current_Unit,
                                        Output_Name      => Value (Option            => Which_O_Option,
                                                                   Explicit_Required => True),
                                        Overwrite_Option => Overwrite_Option);
            Overwrite_Option := False;  -- following units are appended

         when Instantiate =>
            Tools.Instantiator.Process (Unit_Name           => Units_List.Current_Unit,
                                        Unit_Rank           => Unit_Count,
                                        Process_Spec        => Spec_Option,
                                        Process_Body        => Body_Option,
                                        Print_Changed_Lines => Comment_Option,
                                        Output_Prefix       => Value (Option            => Which_O_Option,
                                                                      Explicit_Required => True),
                                        Line_Length         => Line_Length,
                                        Keep_Unchanged      => Which_O_Option = 'O',
                                        Overwrite_Option    => Overwrite_Option);

         when Unrepresent =>
            Tools.Unrepresenter.Process (Unit_Name           => Units_List.Current_Unit,
                                         Process_Spec        => Spec_Option,
                                         Process_Body        => Body_Option,
                                         Print_Changed_Lines => Comment_Option,
                                         Output_Prefix       => Value (Option            => Which_O_Option,
                                                                       Explicit_Required => True),
                                         Line_Length         => Line_Length,
                                         Keep_Unchanged      => Which_O_Option = 'O',
                                         Overwrite_Option    => Overwrite_Option);
         when Unuse =>
            Tools.Unuser.Process (Unit_Name           => Units_List.Current_Unit,
                                  Process_Spec        => Spec_Option,
                                  Process_Body        => Body_Option,
                                  Print_Changed_Lines => Comment_Option,
                                  Output_Prefix       => Value (Option            => Which_O_Option,
                                                                Explicit_Required => True),
                                  Line_Length         => Line_Length,
                                  Keep_Unchanged      => Which_O_Option = 'O',
                                  Overwrite_Option    => Overwrite_Option);
         when Translate =>
            Tools.Translator.Process (Unit_Name           => Units_List.Current_Unit,
                                      Process_Spec        => Spec_Option,
                                      Process_Body        => Body_Option,
                                      Print_Changed_Lines => Comment_Option,
                                      Output_Prefix       => Value (Option            => Which_O_Option,
                                                                    Explicit_Required => True),
                                      Line_Length         => Line_Length,
                                      Keep_Unchanged      => Which_O_Option = 'O',
                                      Overwrite_Option    => Overwrite_Option);
      end case;
      Units_List.Skip;
   end loop;
   User_Log ("Execution time: " & Format_Duration (Clock - Start_Time));

   --
   -- Clean up
   --

   Tools.Finalize;

exception
   when Occur : Options_Error | Units_List.Specification_Error =>
      User_Message (To_Wide_String (Ada.Exceptions.Exception_Message (Occur)));
      User_Message ("try -h for help");

   when Asis.Exceptions.ASIS_Failed =>
      case Status is
         when Asis.Errors.Use_Error =>
            User_Message ("Inconsistent tree, please remove *.adt files");
         when others =>
            Asis_Exception_Messages;
            Print_Help (Version_Only => True);
            User_Log ("Total execution time: " & Format_Duration (Clock - Start_Time));
            raise;
      end case;

   when Occur :
        ASIS_Inappropriate_Context          |
        ASIS_Inappropriate_Container        |
        ASIS_Inappropriate_Compilation_Unit |
        ASIS_Inappropriate_Element          |
        ASIS_Inappropriate_Line             |
        ASIS_Inappropriate_Line_Number
      =>
      declare
         use Ada.Command_Line;
      begin
         Asis_Exception_Messages;
         Stack_Traceback (Occur);
         Print_Help (Version_Only => True);
         Set_Exit_Status (Ada.Command_Line.Failure);
         User_Log ("Total execution time: " & Format_Duration (Clock - Start_Time));
         if Debug_Option then
            raise;  -- To get stack traceback (if compiled with -E at bind)
         end if;
      end;

   when Dictionary.Invalid_Dictionary =>
      User_Message ("Invalid dictionary");

   when Occur : others =>
      declare
         use Ada.Command_Line, Ada.Exceptions;
      begin
         User_Message ("Internal error: " & To_Wide_String (Exception_Name (Occur)));
         User_Message ("       Message: " & To_Wide_String (Exception_Message (Occur)));
         Stack_Traceback (Occur);
         Print_Help (Version_Only => True);
         Set_Exit_Status (Ada.Command_Line.Failure);
         User_Log ("Total execution time: " & Format_Duration (Clock - Start_Time));
         if Debug_Option then
            raise;  -- To get stack traceback (if compiled with -E at bind)
         end if;
      end;
end Adasubst;
