----------------------------------------------------------------------
--  Tools.Instantiator - Package body                               --
--  Copyright (C) 2002, 2017 Adalog                                 --
--  Author: J-P. Rosen                                              --
--                                                                  --
--  ADALOG   is   providing   training,   consultancy,   expertise, --
--  assistance and custom developments  in Ada and related software --
--  engineering techniques.  For more info about our services:      --
--  ADALOG                   Tel: +33 1 41 24 31 40                 --
--  19-21 rue du 8 mai 1945  Fax: +33 1 41 24 07 36                 --
--  94110 ARCUEIL            E-m: info@adalog.fr                    --
--  FRANCE                   URL: http://www.adalog.fr              --
--                                                                  --
--  This  unit is  free software;  you can  redistribute  it and/or --
--  modify  it under  terms of  the GNU  General Public  License as --
--  published by the Free Software Foundation; either version 2, or --
--  (at your  option) any later version.  This  unit is distributed --
--  in the hope  that it will be useful,  but WITHOUT ANY WARRANTY; --
--  without even the implied warranty of MERCHANTABILITY or FITNESS --
--  FOR A  PARTICULAR PURPOSE.  See the GNU  General Public License --
--  for more details.   You should have received a  copy of the GNU --
--  General Public License distributed  with this program; see file --
--  COPYING.   If not, write  to the  Free Software  Foundation, 59 --
--  Temple Place - Suite 330, Boston, MA 02111-1307, USA.           --
--                                                                  --
--  As  a special  exception, if  other files  instantiate generics --
--  from  this unit,  or you  link this  unit with  other  files to --
--  produce an executable,  this unit does not by  itself cause the --
--  resulting executable  to be covered  by the GNU  General Public --
--  License.  This exception does  not however invalidate any other --
--  reasons why  the executable  file might be  covered by  the GNU --
--  Public License.                                                 --
----------------------------------------------------------------------

with   -- Standard Ada units
   Ada.Characters.Handling,
   Ada.Strings.Wide_Fixed,
   Ada.Strings.Wide_Maps,
   Ada.Strings.Wide_Unbounded,
   Ada.Wide_Text_IO;

with   -- ASIS components
   Asis.Clauses,
   Asis.Compilation_Units,
   Asis.Declarations,
   Asis.Elements,
   Asis.Expressions,
   Asis.Iterator,
   Asis.Text;
pragma Elaborate_All (Asis.Iterator);

with   -- Adalog components
   Producer,
   Scope_Manager,
   Thick_Queries,
   Units_List,
   Utilities;
package body Tools.Instantiator is
   use Ada.Strings.Wide_Unbounded;
   use Asis, Asis.Elements;
   package ASW renames Ada.Strings.Wide_Maps;

   -- Principle

   -- The instantiations must preserve visibilities, and not introduce visibilities to more entities that could
   -- clash with the rest of the code. They must not duplicate evaluations of elements when there is only one for
   -- a real instantiations.
   -- We accept to create one extra entity if its name can be considered highly unlikely to create conflicts.

   -- Case of nested (non-library) instantiations
   -- -------------------------------------------
   -- A wrapper package is introduced (whose name is the name of the instantiation & "_Instantion_Wrapper", unlikely
   -- to happen in the rest of the code), that contains renaming declarations whose names are those of the formal,
   -- and whose renamed entities are the corresponding actual. "in" parameters are declared as constants rather than
   -- renamings, because some compilers seem to have issues with renamings of expressions.
   --
   -- Then, the generic package is copied verbatim into the wrapper package (of course, the body of the generic is
   -- copied into the body of the wrapper package).
   --
   -- Finally, after the package, a renaming declaration declares the instance as a renaming of the copy of the
   -- generic from the wrapper package.
   --
   -- Case of library instantiations
   -- ------------------------------
   -- The above strategy does not work for library instantiations, because a library renaming must target a library
   -- unit, and therefore cannot target the copy inside the wrapper.
   --
   -- Case of library package instantiations:
   -- Here, the wrapper package contains only the declarations, and the generic is copied outside of the wrapper,
   -- as a library unit, changing its name to the instance name (and of course, every reference inside the generic
   -- to its own name has to be patched accordingly).
   -- A "with" clause and a "use" clause for the wrapper package are added on top of that unit to ensure visibility
   -- of parameters.
   --
   -- This creates three compilation units (the wrapper package specification - it has no body -, the specification
   -- and body of the instantiated unit) in the same output file. With gnat, a run of gnatchop is necessary.
   --
   -- Note that this strategy cannot be used for non-library instantiations, because the use clause would extend
   -- beyond the instantiation and create potential conflicts.
   --
   -- Case of library subprogram instantiations:
   --
   --
   -- Annoying case
   -- -------------
   -- When a generic is instantiated inside a package spec, the body of the instance appears logically at the place of
   -- the instantiation. But we can't have an explicit body in a package spec!
   -- The best we can do is to move the instantiated body at the beginning of the body of the enclosing package - and
   -- of course, such a body must be generated if there was none initially.

   ----------------------------------------------------------------
   --                     Parameters                             --
   ----------------------------------------------------------------
   Max_Generic_Depth : constant := 10; -- A reasonable maximum for the depth of:
                                       -- - Nested generics
                                       -- - Nested instantiations
                                       -- - Child generic units
                                       -- - Nesting of generics in package specs

   ----------------------------------------------------------------
   --                 Internal elements                          --
   ----------------------------------------------------------------
   Fixed_Name          : constant Wide_String := "_Instantiation_Wrapper";
   Instantiation_Count : Natural;    -- For displaying the number of instantiations in the current unit

   Expanded_Name_Fix : constant ASW.Wide_Character_Mapping := ASW.To_Mapping (".", "_");

   -- State info for the general traversal of generics
   type Traverse_Info is
      record
         Duplicate    : Boolean;                -- True if we are duplicating a generic (otherwise, skip the generic)
         Def_Name     : Asis.Defining_Name;     -- Defining name of generic being traversed
         Inst_Pos     : Natural;                -- Position number of the instantiation, used for key generation
         Wrapper_Name : Unbounded_Wide_String;  -- Name of the wrapper package surrounding the duplication (or null UWS)
         Copy_Name    : Unbounded_Wide_String;  -- Name given to the duplicate
      end record;

   Instantiation_Depth   : ASIS_Natural := 0;  -- Depth of nested instantiations
   Active_Instantiations : Asis.Element_List (1 .. Max_Generic_Depth);


   type Instantiation_Context is
      record
         Specs_Count   : ASIS_Natural := 0;                     -- All pack. specs without bodies surrounding the
         Encl_Specs    : Element_List (1 .. Max_Generic_Depth); -- instanciation
         Instantiation : Asis.Declaration;                      -- The instantiation that generates this wrapper
         T_Info        : Traverse_Info;
      end record;

   -- The scoped store for delayed wrappers bodies:
   -- Since we traverse the same generic several times (for each instantiation), we use the name of the instantiation
   -- rather than the name of the generic as a key to keep the info between spec and body
   function Instantiation_Key (Scope : Asis.Element) return Ada.Strings.Wide_Unbounded.Unbounded_Wide_String;
   package Delayed_Wrappers_Store is new Scope_Manager.Scoped_Store (Instantiation_Context,
                                                                     Scope_Key => Instantiation_Key);

   ---------------------
   -- Simple_Def_Name --
   ---------------------

   function Simple_Def_Name (Def_Name : Asis.Defining_Name) return Asis.Defining_Name is
   -- Like Thick_Queries.Simple_Name, but for defining names
      use Asis.Declarations;
   begin
      if Defining_Name_Kind (Def_Name) = A_Defining_Expanded_Name then
         return Defining_Selector (Def_Name);
      else
         return Def_Name;
      end if;
   end Simple_Def_Name;

   -----------------------
   -- Instantiation_Key --
   -----------------------

   Current_Instantiation_Pos : Natural;
   function Instantiation_Key (Scope : Asis.Element) return Ada.Strings.Wide_Unbounded.Unbounded_Wide_String is
      use Scope_Manager, Thick_Queries;
   begin
      if Is_Generic_Unit (Scope) then
         return To_Unbounded_Wide_String (Natural'Wide_Image (Current_Instantiation_Pos)) & Default_Key (Scope);
      else
         return Default_Key (Scope);
      end if;
   end Instantiation_Key;

   --------------
   -- Traverse --
   --------------

   procedure Pre_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Traverse_Info);
   procedure Post_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Traverse_Info);
   procedure Traverse is new Asis.Iterator.Traverse_Element (Traverse_Info, Pre_Procedure, Post_Procedure);

   -------------------
   -- Pre_Procedure --
   -------------------

   -- Forward declarations
   procedure Generate_Wrapper_Body (Wrapper_Name : Wide_String;
                                    Generic_Spec : Asis.Declaration;
                                    T_Info       : Traverse_Info);
   procedure Generate_Delayed_Wrapper (For_Instantiation : in Instantiation_Context);
   procedure Finalize_Delayed_Wrappers;
   procedure Process_Instantiation (Inst : Asis.Declaration);
   procedure Duplicate_Generic_Part (Part : Asis.Declaration; T_Info: Traverse_Info);

   procedure Pre_Procedure (Element : in     Asis.Element;
                            Control : in out Asis.Traverse_Control;
                            State   : in out Traverse_Info)
   is
      use Asis.Declarations, Asis.Expressions;
      use Producer, Scope_Manager, Thick_Queries;

      procedure Process_Identifier_Path (Name : Asis.Element_List) is
      -- If any part of the selectors of Name is a reference to the (previous) name of the generic, replace it and
      -- all preceding names by the name of the copy, qualified by the name of the wrapper if any (the wrapper is
      -- always visible from the copy). If no wrapper, no qualification is needed since the copy is a library unit.
      --
      -- Otherwise, if the last name (the selector) refers to an element declared outside the generic, replace the
      -- whole name by its full name, since the visibility at the point of instantiation is totally different from
      -- the visibility inside the generic.
         Clean_Name : Asis.Expression;
      begin
         for N : Asis.Element of Name loop
            Clean_Name := N;
            loop
               case Expression_Kind (Clean_Name) is
                  when An_Explicit_Dereference | An_Indexed_Component =>
                     Clean_Name := Prefix (Clean_Name);
                  when A_Function_Call =>
                     Clean_Name := Simple_Name (Prefix (Clean_Name));
                  when others =>
                     exit;
               end case;
            end loop;

            if Is_Equal (Simple_Def_Name (First_Defining_Name (Clean_Name)), State.Def_Name) then
               Print_Up_To (Name (Name'First), Included => False);
               if State.Wrapper_Name = Null_Unbounded_Wide_String then
                  Print (To_Wide_String (State.Copy_Name));
               else
                  Print (To_Wide_String (State.Wrapper_Name) & '.' & To_Wide_String (State.Copy_Name));
               end if;
               Advance (Clean_Name, Included => True);
               return;
            end if;
         end loop;

         -- Here, no selector was a reference to the name of the generic

         if Declaration_Kind (Corresponding_Name_Declaration (Name (Name'Last))) in A_Formal_Declaration then
            -- Formal elements are visible only within the generic, therefore they are still visible here
            -- => no qualification needed (and it avoids trouble in the case of a nested generic refering to the
            --    formals of an enclosing generic)
            return;
         end if;

         if Declaration_Kind (Corresponding_Name_Declaration (Name (Name'Last))) = A_Parameter_Specification then
            -- Same thing for formal parameters of callable entities
            return;
         end if;

         declare
            use Asis.Compilation_Units;
            use Utilities;

            Gen_Spec  :          Asis.Declaration := Enclosing_Element (State.Def_Name);
            Gen_Body  :          Asis.Declaration;
            Elem_Decl : constant Asis.Declaration := Corresponding_Name_Declaration (Name (Name'Last));
         begin
            -- Eliminate the case of entities declared in Standard
            -- (would not harm, but it looks ugly to generate "Standard." for these
            if Is_Nil (Elem_Decl) -- some predefined operator...
              or else To_Upper (Unit_Full_Name (Enclosing_Compilation_Unit (Elem_Decl))) = "STANDARD"
            then
               return;
            end if;

            while Declaration_Kind (Gen_Spec) not in A_Generic_Declaration loop
               -- We can have defining expanded names
               Gen_Spec := Enclosing_Element (Gen_Spec);
            end loop;
            Gen_Body := Corresponding_Body (Gen_Spec);
            if not Is_Part_Of (Elem_Decl, Gen_Spec)
              and then (Is_Nil (Gen_Body) or else not Is_Part_Of (Elem_Decl, Gen_Body))
            then
               Print_Up_To (Name (Name'First), Included => False);
               Print (Full_Name_Image (Name (Name'Last)));
               Advance (Name (Name'Last), Included => True);
            end if;
         end;
      end Process_Identifier_Path;

      Temp : Asis.Element;
   begin   -- Pre_Procedure
      if Is_Scope (Element) then
         Scope_Manager.Enter_Scope (Element);
      end if;

      case Element_Kind (Element) is
         when A_Clause =>
            if Is_Part_Of_Generic (Element) then
               -- Ignore clauses (use clauses) that are within the formal parameters part
               Control := Abandon_Children;
            end if;

         when A_Declaration =>
            case Declaration_Kind (Element) is
               when A_Generic_Instantiation =>
                  Process_Instantiation (Element);
                  Control := Abandon_Children;
                  Scope_Manager.Exit_Scope (Element);
               when A_Generic_Declaration =>
                  if State.Duplicate then
                     State.Duplicate := False;   -- For nested generics

                     Temp := Names (Element) (1);
                     Print_Up_To (Temp, Included => False);
                     if Is_Equal (Simple_Def_Name (First_Defining_Name (Simple_Def_Name(Temp))), State.Def_Name) then
                        -- This is the defining name of the generic, replace with Copy_Name
                        Print (To_Wide_String (State.Copy_Name));
                        Advance (Temp, Included => True);
                     end if;
                  else
                     -- remove it
                     Print_Up_To (Element, Included => False);
                     Advance     (Element, Included => True);
                     Control := Abandon_Children;
                     Scope_Manager.Exit_Scope (Element);
                  end if;
               when A_Procedure_Body_Declaration | A_Function_Body_Declaration =>
                  if Is_Generic_Unit (Element) and not State.Duplicate then
                     -- Remove the generic
                     Print_Up_To (Element, Included => False);
                     Advance     (Element, Included => True);
                     Control := Abandon_Children;
                     Scope_Manager.Exit_Scope (Element);
                  else   -- Duplicating, or not a generic
                     State.Duplicate := False;   -- For nested generics
                  end if;
               when A_Package_Body_Declaration =>
                  if Is_Generic_Unit (Element) and not State.Duplicate then
                     -- Remove the generic
                     Print_Up_To (Element, Included => False);
                     Advance     (Element, Included => True);
                     Control := Abandon_Children;
                     Scope_Manager.Exit_Scope (Element);
                  else   -- Duplicating, or not a generic
                     State.Duplicate := False;   -- For nested generics

                     -- The following is harmless if we have no delayed bodies (and necessary otherwise)
                     Temp := Names (Element) (1);
                     Print_Up_To (Temp, Included => False);
                     if Is_Equal (Simple_Def_Name (First_Defining_Name (Simple_Def_Name (Temp))), State.Def_Name) then
                           -- This is the defining name of the generic, replace with Copy_Name
                        Print (To_Wide_String (State.Copy_Name));
                        Advance (Temp, Included => True);
                     else
                        Print_Up_To (Temp, Included => True);
                     end if;
                     Print_Up_To ("IS", Included => True, Ref_Elem => Element);

                     -- Do we have delayed bodies to generate?
                     declare
                        -- Careful with iterators! This is called recursively (but on different slices of the scope
                        -- stack)
                        Save_Curs : constant Delayed_Wrappers_Store.Cursor := Delayed_Wrappers_Store.Current_Cursor;
                     begin
                        Delayed_Wrappers_Store.Reset (Current_Scope_Only);
                        if Delayed_Wrappers_Store.Data_Available then
                           while Delayed_Wrappers_Store.Data_Available loop
                              Generate_Delayed_Wrapper (Delayed_Wrappers_Store.Current_Data);
                              Delayed_Wrappers_Store.Delete_Current;
                           end loop;
                           Finalize_Delayed_Wrappers;
                        end if;
                        Delayed_Wrappers_Store.Restore (Save_Curs);
                     end;
                  end if;
               when A_Formal_Declaration =>
                  -- Ignored when duplicating (and should not be traversed, because they contain identifiers)
                  Control := Abandon_Children;
                  if Is_Scope (Element) then
                     Scope_Manager.Exit_Scope (Element);
                  end if;
               when others =>
                  null;
            end case;

         when A_Defining_Name =>
            -- The defining name of a copy is handled when entering the generic spec/body
            -- Identifiers that are part of a defining expanded name should not be processed.
            Control := Abandon_Children;

         when An_Expression =>
            case Expression_Kind (Element) is
               when An_Identifier =>
                  if Is_Part_Of_Generic (Element) then
                     -- We can be inside a generic only when duplicating
                     Process_Identifier_Path ((1 => Element));
                  end if;
               when A_Selected_Component =>
                  if Is_Part_Of_Generic (Element) then
                     -- Eliminate the case of a selected component which is part of a defining expanded name
                     Temp := Enclosing_Element (Element);
                     while Expression_Kind (Temp) = A_Selected_Component loop
                        Temp := Enclosing_Element (Temp);
                     end loop;
                     if Defining_Name_Kind (Temp) /= A_Defining_Expanded_Name then
                        Process_Identifier_Path (Name_Path (Element));
                     end if;
                     -- Don't recurse in the selected name to avoid double processing of identifiers.
                     Control := Abandon_Children;
                  end if;
               when An_Attribute_Reference =>
                  -- Traverse the prefix only
                  Traverse (Prefix (Element), Control, State);
                  Control := Abandon_Children;
               when others =>
                  null;
            end case;

        when A_Pragma =>
            -- Nothing interesting for us here (until proven otherwise), and those are full of exotic identifiers...
            Control := Abandon_Children;

         when others =>
            null;
      end case;
   end Pre_Procedure;

   --------------------
   -- Post_Procedure --
   --------------------

   procedure Post_Procedure (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                             State   : in out Traverse_Info)
   is
      pragma Unreferenced (Control);
      use Asis.Declarations;
      use Producer, Scope_Manager, Thick_Queries;

      End_Name  : Asis.Expression;

      procedure Create_Package_Body is
      -- Creates a body for a bodyless package that now requires one due to instantiations in the specification
      -- The body is created here (after the spec), unless the enclosing scope is itself a specification, in which
      -- case it has to be delayed to the enclosing level
         use Utilities;

         Pack_Instance : Instantiation_Context;
         Old_Curs      : constant Delayed_Wrappers_Store.Cursor := Delayed_Wrappers_Store.Current_Cursor;
      begin
         Delayed_Wrappers_Store.Reset (Current_Scope_Only);
         if not Delayed_Wrappers_Store.Data_Available then
            -- no attached bodies, not necessary to generate body
            Delayed_Wrappers_Store.Restore (Old_Curs);
            return;
         end if;

         if Declaration_Kind (Current_Scope) in A_Generic_Declaration  then
            -- The true enclosing scope is the wrapper spec. Can't generate the body here, but wait till the wrapper
            -- is generated. Keep it attached to the current scope
            null;
         elsif Declaration_Kind (Enclosing_Scope) in A_Package_Declaration | A_Generic_Package_Declaration then
            -- Can't generate the body here either... Delay to next level
            while Delayed_Wrappers_Store.Data_Available loop
               Pack_Instance := Delayed_Wrappers_Store.Current_Data;
               if Pack_Instance.Specs_Count = Max_Generic_Depth then
                  Failure ("Too deeply nested instantiation, maximum is " & ASIS_Integer_Img (Max_Generic_Depth));
               end if;
               Pack_Instance.Specs_Count := Pack_Instance.Specs_Count + 1;
               Pack_Instance.Encl_Specs (2 .. Pack_Instance.Specs_Count)
                 := Pack_Instance.Encl_Specs (1 .. Pack_Instance.Specs_Count - 1);
               Pack_Instance.Encl_Specs (1) := Current_Scope;
               Delayed_Wrappers_Store.Prepend_Enclosing (Pack_Instance);
               Delayed_Wrappers_Store.Delete_Current;
            end loop;
         else
            Next_Line;
            Print_Line ("package body " & Defining_Name_Image (Names (Element)(1)) & " is");
            if Delayed_Wrappers_Store.Data_Available then
               while Delayed_Wrappers_Store.Data_Available loop
                  Generate_Delayed_Wrapper (Delayed_Wrappers_Store.Current_Data);
                  Delayed_Wrappers_Store.Delete_Current;
               end loop;
               Finalize_Delayed_Wrappers;
            end if;
            Print_Line ("end " & Defining_Name_Image (Names (Element)(1)) & ';');
         end if;

         Delayed_Wrappers_Store.Restore (Old_Curs);
      end Create_Package_Body;

   begin   -- Post_Procedure
      case Declaration_Kind (Element) is
         when A_Package_Declaration =>
            -- if the package has no body and there were instantiations in the spec, we must create a body
            -- (but not necessarily here)
            Print_Up_To (Element, Included => True);
            if Is_Nil (Corresponding_Body (Element)) then
               Create_Package_Body;
            end if;
         when A_Generic_Package_Declaration =>
            -- must fix the identifier after "end"
            End_Name := Corresponding_End_Name (Element);
            if not Is_Nil (End_Name) then   -- Fix the name at the end of duplication
               Print_Up_To (End_Name, Included => False);
               Print (To_Wide_String (State.Copy_Name));
               Advance (End_Name, Included => True);
               Print_Up_To (Element, Included => True);  -- For the final ';'
            end if;

            -- if the package has no body and there were instantiations in the spec, we must create a body
            -- (otherwise, the generic bodies will be produced on entering the package body)
            if Is_Nil (Corresponding_Body (Element)) then
               Create_Package_Body;
            end if;
         when A_Procedure_Body_Declaration
            | A_Function_Body_Declaration
            | A_Package_Body_Declaration
            =>
            End_Name := Corresponding_End_Name (Element);
            if Is_Generic_Unit (Element) and then not Is_Nil (End_Name) then -- Fix the name at the end of duplication
               Print_Up_To (End_Name, Included => False);
               Print (To_Wide_String (State.Copy_Name));
               Advance (End_Name, Included => True);
            end if;
         when others =>
            null;
      end case;

      if Scope_Manager.Is_Scope (Element) then
         Scope_Manager.Exit_Scope (Element);
      end if;
   end Post_Procedure;

   ---------------------------
   -- Generate_Wrapper_Body --
   ---------------------------

   procedure Generate_Wrapper_Body (Wrapper_Name : Wide_String;
                                    Generic_Spec : Asis.Declaration;
                                    T_Info       : Traverse_Info)
   is
      use Producer;
      use Asis.Declarations, Asis.Text;

      Generic_Body : constant Asis.Declaration := Corresponding_Body (Generic_Spec);
      Generic_Span : constant Span := Element_Span (Generic_Body);
      Subpack_Curs : Delayed_Wrappers_Store.Cursor;
      Old_Pos      : Natural;
   begin
      if Is_Nil (Generic_Body) then
         -- We must still generate a body for the (instantiated) generic if the spec contains
         -- instantiations that require a body
         Old_Pos := Current_Instantiation_Pos;
         Current_Instantiation_Pos := T_Info.Inst_Pos;
         Delayed_Wrappers_Store.Create_Cursor (Subpack_Curs, On => Generic_Spec);
         Current_Instantiation_Pos := Old_Pos;
         if Delayed_Wrappers_Store.Data_Available (Subpack_Curs) then
            Print_Line ("package body " & Wrapper_Name & " is");
            Print_Line ("", In_Col => Generic_Span.First_Column);

            Print_Line ("package body " & To_Wide_String (T_Info.Copy_Name) & " is");
            while Delayed_Wrappers_Store.Data_Available (Subpack_Curs) loop
               Generate_Delayed_Wrapper (Delayed_Wrappers_Store.Current_Data (Subpack_Curs));
               Delayed_Wrappers_Store.Next (Subpack_Curs);
            end loop;
            Finalize_Delayed_Wrappers;
            Print_Line ("end " & To_Wide_String (T_Info.Copy_Name) & ';');
            Print_Line ("end " & Wrapper_Name & ";");
         end if;
      else
         Print_Line ("package body " & Wrapper_Name & " is");
         Next_Line;
         Duplicate_Generic_Part (Generic_Body, T_Info);
         Print_Line ("end " & Wrapper_Name & ";");
         Next_Line;
      end if;
   end Generate_Wrapper_Body;


   ------------------------------
   -- Generate_Delayed_Wrapper --
   ------------------------------

   Open_Specs : Element_List (1 .. Max_Generic_Depth);
   Open_Count : ASIS_Natural := 0;
   procedure Generate_Delayed_Wrapper (For_Instantiation : in Instantiation_Context) is
   -- Generate the delayed wrappers bodies for the current scope, taking care of possible enclosing extra packages
      use Asis.Declarations, Asis.Expressions;
      use Thick_Queries, Producer;
   begin
      -- Close open packages if not equal to current
      while Open_Count > 0
        and then (For_Instantiation.Specs_Count = 0
                  or else not Is_Equal (Open_Specs (Open_Count),
                                        For_Instantiation.Encl_Specs (Open_Count)))
      loop
         Print_Line ("end " & Defining_Name_Image (Names (Open_Specs (Open_Count)) (1)) & ';');
         Open_Count := Open_Count - 1;
      end loop;

      -- Open possible extra packages
      Open_Specs := For_Instantiation.Encl_Specs;
      for Spec : Asis.Element of Open_Specs (Open_Count + 1 .. For_Instantiation.Specs_Count) loop
         Print_Line ("package body " & Defining_Name_Image (Names (Spec) (1)) & " is");
      end loop;
      Open_Count := For_Instantiation.Specs_Count;

      Generate_Wrapper_Body (Wrapper_Name => Defining_Name_Image (Names (For_Instantiation.Instantiation) (1))
                                             & Fixed_Name,
                             Generic_Spec => Corresponding_Name_Declaration (Simple_Name
                                                                             (Generic_Unit_Name
                                                                              (For_Instantiation.Instantiation))),
                             T_Info       => For_Instantiation.T_Info);

   end Generate_Delayed_Wrapper;


   -------------------------------
   -- Finalize_Delayed_Wrappers --
   -------------------------------

   procedure Finalize_Delayed_Wrappers is
   -- close remaining open packages
   use Asis.Declarations;
   use Producer;
   begin
      for Spec : Asis.Element of reverse Open_Specs (1 .. Open_Count) loop
         Print_Line ("end " & Defining_Name_Image (Names (Spec) (1)) & ';');
      end loop;
      Open_Count := 0;
   end Finalize_Delayed_Wrappers;

   ----------------------------
   -- Duplicate_Generic_Part --
   ----------------------------

   procedure Duplicate_Generic_Part (Part : Asis.Declaration; T_Info : Traverse_Info) is
      use Asis.Declarations, Asis.Text;
      use Producer;

      Good_Part      : Asis.Declaration := Part;
      The_Info       : Traverse_Info := T_Info;   -- Because we need a variable for Traverse
      Previous_State : Producer.State;
      Old_Curs       : constant Delayed_Wrappers_Store.Cursor := Delayed_Wrappers_Store.Current_Cursor;
      Generic_Span   : Span;
      The_Control    : Traverse_Control := Continue;
      Old_Pos        : constant Natural := Current_Instantiation_Pos;
   begin
      Current_Instantiation_Pos := The_Info.Inst_Pos;
      while Is_Part_Of_Instance (Good_Part) loop
         -- Generic from an instance of a generic => get source from the ultimate generic
         Good_Part := Enclosing_Element (Corresponding_Generic_Element (Names (Good_Part)(1)));
      end loop;
      Generic_Span := Element_Span (Good_Part);
      if Declaration_Kind (Good_Part) in A_Generic_Declaration then  -- i.e. not a generic body
         -- Find the start of the procedure/function/package specification, after the generic formal part
         declare
            Formals : constant Asis.Element_List := Generic_Formal_Part (Good_Part, Include_Pragmas => True);
         begin
            if Is_Nil (Formals) then
               Generic_Span.First_Column := Generic_Span.First_Column + Wide_String'("generic")'Length;
            else
               Generic_Span.First_Line   := Element_Span (Formals (Formals'Last)).Last_Line;
               Generic_Span.First_Column := Element_Span (Formals (Formals'Last)).Last_Column + 1;
            end if;
         end;
      end if;

      Next_Line (Conditional => True);
      Push_Source (Previous_State, Generic_Span.First_Line, Generic_Span.First_Column);

      Traverse (Part, The_Control, The_Info);
      Print_Up_To (Part, Included => True);
      Next_Line;

      Pop_Source (Previous_State);
      Delayed_Wrappers_Store.Restore (Old_Curs);
      Current_Instantiation_Pos := Old_Pos;
   end Duplicate_Generic_Part;

   ---------------------------
   -- Process_Instantiation --
   ---------------------------

   procedure Process_Instantiation (Inst : Asis.Declaration) is
      use Ada.Strings.Wide_Fixed;
      use Asis.Declarations, Asis.Expressions, Asis.Text;
      use Producer, Thick_Queries, Utilities;

      Instance_Name  : constant Wide_String := Defining_Name_Image (Names (Inst) (1));
      Wrapper_Name   : constant Wide_String := Instance_Name & Fixed_Name;
      Generic_Name   : constant Wide_String := Translate (Defining_Name_Image (Corresponding_Name_Definition
                                                                               (Simple_Name
                                                                                (Generic_Unit_Name (Inst)))),
                                                         Expanded_Name_Fix);
      Actuals        : constant Asis.Association_List := Generic_Actual_Part (Inst, Normalized => True);
      Generic_Spec   : constant Asis.Declaration      := Corresponding_Name_Declaration (Simple_Name
                                                                                         (Generic_Unit_Name (Inst)));
      Generic_Body   : constant Asis.Declaration      := Corresponding_Body (Generic_Spec);
      Generic_Span   : Span;

      Previous_State : State;
      The_Info       : Traverse_Info;

      Formals        : constant Asis.Element_List := Generic_Formal_Part (Generic_Spec, Include_Pragmas => True);

      procedure Print_Fixed_For_Formals (Elem : Asis.Element) is
         -- Prints Elem, but if it is a generic formal, prefix it with the name of the wrapper package.
         Elem_Span : constant Span := Element_Span (Elem);
         Old_State : State;

         type Null_Info is null record;
         The_Control : Traverse_Control := Continue;
         Unused_Info : Null_Info;

         procedure Pre_Procedure (Element : in     Asis.Element;
                                  Control : in out Asis.Traverse_Control;
                                  State   : in out Null_Info)
         is
            pragma Unreferenced (Control, State);

            function Corresponding_Wrapper_Name (Elem_Decl : Asis.Declaration) return Wide_String is
               Good_Decl : Asis.Declaration := Elem_Decl;
            begin
               while Declaration_Kind (Good_Decl) not in A_Generic_Declaration loop
                  Good_Decl := Enclosing_Element (Good_Decl);
               end loop;
               for Active_Inst : Asis.Element of reverse Active_Instantiations (1 .. Instantiation_Depth) loop
                  -- Reverse loop because it is more likely to be local
                  -- Micro-optimization? Yes, but it costs nothing...
                  if Is_Equal (Corresponding_Name_Declaration (Simple_Name (Generic_Unit_Name (Active_Inst))),
                               Good_Decl)
                  then
                     return Defining_Name_Image (Names (Active_Inst)(1)) & Fixed_Name;
                  end if;
               end loop;
               Failure ("Not found in instantiation stack", Elem_Decl);
            end Corresponding_Wrapper_Name;

            Decl : Asis.Declaration;
         begin   -- Pre_Procedure
            case Expression_Kind (Element) is
               when An_Identifier =>
                  Print_Up_To (Element, Included => False);
                  Decl := Corresponding_Name_Declaration (Element);
                  if Declaration_Kind (Decl) in A_Formal_Declaration then
                     Print (Corresponding_Wrapper_Name (Decl) & '.');
                  end if;
                  Print_Up_To (Element, Included => True);
               when others =>
                  null;
            end case;
         end Pre_Procedure;

         procedure Post_Procedure (Element : in Asis.Element;
                                   Control : in out Asis.Traverse_Control;
                                   State   : in out Null_Info) is null;
         procedure Traverse_For_Fix is new Asis.Iterator.Traverse_Element (Null_Info, Pre_Procedure, Post_Procedure);

      begin   -- Print_Fixed_For_Formals
         Push_Source (Old_State, Elem_Span.First_Line, Elem_Span.First_Column);
         Traverse_For_Fix (Elem, The_Control, Unused_Info);
         Print_Up_To (Elem, Included => True);
         Pop_Source (Old_State);
      end Print_Fixed_For_Formals;

      procedure Print_Fixed_For_Formals (List : Asis.Parameter_Specification_List) is
      begin
         if Is_Nil (List) then
            return;
         end if;

         Print (" (");
         Print_Fixed_For_Formals (List (List'First));
         for Param : Asis.Parameter_Specification of List (List'First + 1 .. List'Last) loop
            Print ("; ");
            Print_Fixed_For_Formals (Param);
         end loop;
         Print (")");
      end Print_Fixed_For_Formals;

      F_Inx : List_Index;

      use Scope_Manager;
   begin   -- Process_Instantiation
      -- Do nothing for instantiations of language defined generics
      if Compilation_Units.Unit_Origin (Enclosing_Compilation_Unit (Generic_Spec)) /= An_Application_Unit then
         return;
      end if;

      Instantiation_Count       := Instantiation_Count + 1;
      Current_Instantiation_Pos := Instantiation_Count;

      if Instantiation_Depth = Max_Generic_Depth then
         Failure ("Too many nested instantiations, maximum is " & ASIS_Integer_Img (Max_Generic_Depth));
      end if;
      Instantiation_Depth := Instantiation_Depth + 1;
      Active_Instantiations (Instantiation_Depth) := Inst;

      Print_Up_To (Inst, Included => False);

      --
      -- Wrapper package spec header
      --
      Next_Line;
      Print_Line ("package " & Wrapper_Name & " is");

      --
      -- Generic actuals
      --

      -- Generate renamings (or constant) declarations from the actuals, but scan the formals in parallel to
      -- add possible pragmas or use clauses at the same place
      F_Inx := Formals'First;
      for Assoc : Asis.Association of Actuals loop
         while Declaration_Kind (Formals (F_Inx)) not in A_Formal_Declaration loop
            Print_Line (Trim_All (Element_Image (Formals (F_Inx))));
            F_Inx := F_Inx + 1;
         end loop;
         F_Inx := F_Inx + 1;

         declare
            Formal_Name : constant Asis.Defining_Name := Formal_Parameter (Assoc);
            Actual      : constant Asis.Expression    := Actual_Parameter (Assoc);

            Formal_Decl : constant Asis.Declaration := Enclosing_Element (Formal_Name);
         begin
            case A_Formal_Declaration'(Declaration_Kind (Formal_Decl)) is
               when A_Formal_Object_Declaration =>
                  case Mode_Kind (Formal_Decl) is
                     when An_In_Mode | A_Default_In_Mode =>
                        Print_Line (Trim_All (Element_Image (Formal_Name)) & " : constant "
                                    & Trim_All (Element_Image (Object_Declaration_View (Formal_Decl)))
                                    & " := " & Trim_All (Element_Image (Actual)) & ";");
                     when An_In_Out_Mode =>
                        Print_Line (Trim_All (Element_Image (Formal_Name)) & " : "
                                    & Trim_All (Element_Image (Object_Declaration_View (Formal_Decl)))
                                    & " renames " & Trim_All (Element_Image (Actual)) & ";");
                     when others =>
                        Failure ("Instantiate: Bad mode for formal object", Formal_Decl);
                  end case;
               when A_Formal_Type_Declaration | A_Formal_Incomplete_Type_Declaration =>
                  Print_Line ("subtype " &  Trim_All (Element_Image (Formal_Name)) & " is "
                              & Trim_All (Element_Image (Actual) & ";"));
               when A_Formal_Procedure_Declaration =>
                  declare
                     Proc_Params : constant Parameter_Specification_List := Parameter_Profile (Formal_Decl);
                  begin
                     Print_Line ("procedure " & Defining_Name_Image (Names (Formal_Decl) (1))
                                 & Choose (Is_Nil (Proc_Params),
                                           "",
                                           " (" & Trim_All (Element_Image_List (Proc_Params, ";")) & ")")
                                 & " renames " & Trim_All (Full_Name_Image (Actual)) & ";");
                  end;
               when A_Formal_Function_Declaration =>
                  -- if the function has a <> default, and there is no explicit actual, and the actual resolves to
                  -- a predefined operator, then Actual has no Element_Image (at least in A4G).
                  -- Anyway, the name is necessarilly the same as the formal, but we cannot have a renaming with
                  -- identical names. Since the actual is visible at the point of instantiation, we may as well not
                  -- generate any renaming.
                  -- TBSL : the same may arise each time the actual is identical to the formal, for all formal kinds.
                  --        Removing the declaration would work, except for library instantiations...
                  if Element_Image (Actual) /= "" then
                     declare
                        Func_Params : constant Parameter_Specification_List := Parameter_Profile (Formal_Decl);
                     begin
                        Print_Line ("function " & Defining_Name_Image (Names (Formal_Decl) (1))
                                    & Choose (Is_Nil (Func_Params),
                                              "",
                                              " (" & Trim_All (Element_Image_List (Func_Params, ";")) & ")")
                                    & " return " & Trim_All (Element_Image (Result_Profile (Formal_Decl)))
                                    & " renames " & Trim_All (Element_Image (Actual)) & ";");
                     end;
                  end if;
               when A_Formal_Package_Declaration | A_Formal_Package_Declaration_With_Box =>
                  Print_Line ("package " & Defining_Name_Image (Names (Formal_Decl) (1))
                              & " renames " & Trim_All (Full_Name_Image (Actual)) & ";");
            end case;
         end;
      end loop;
      -- Don't forget extra declarations from formals at the end...
      while F_Inx <= Formals'Last loop   --## rule line off simplifiable_statements ## not really a "for" loop here
         Print_Line (Trim_All (Element_Image (Formals (F_Inx))));
         F_Inx := F_Inx + 1;
      end loop;

      --
      -- Duplicate generic spec
      --
      if not Is_Compilation_Unit (Inst) then
         -- We don't duplicate the generic inside the wrapper for library instantiations
         The_Info := (Duplicate    => True,
                      Def_Name     => Simple_Def_Name (Names (Generic_Spec) (1)),
                      Inst_Pos     => Instantiation_Count,
                      Wrapper_Name => To_Unbounded_Wide_String (Wrapper_Name),
                      Copy_Name    => To_Unbounded_Wide_String (Generic_Name));
         Duplicate_Generic_Part (Generic_Spec, The_Info);
      end if;

      --
      -- Wrapper package spec footer
      --
      Print_Line ("end " & Wrapper_Name & ";");

      --
      -- Wrapper package body, duplicate generic body (unless library package instantiation)
      -- or attach the delayed wrapper to the enclosing scope
      --
      if not Is_Compilation_Unit (Inst) then
         The_Info := (Duplicate    => True,
                      Def_Name     => Simple_Def_Name (Names (Generic_Spec) (1)),
                      Inst_Pos     => Instantiation_Count,
                      Wrapper_Name => To_Unbounded_Wide_String (Wrapper_Name),
                      Copy_Name    => To_Unbounded_Wide_String (Generic_Name));
         if Declaration_Kind (Enclosing_Scope) in  A_Package_Declaration | A_Generic_Package_Declaration then
            -- Ouch! Can't put body here
            -- Case of Generic_Package_Declaration:
            -- this happens only when we traverse for duplication. But since we use a normal traversal for
            -- the duplication, the scope manager handles it normally, and the body will be recognized as it
            -- should be.
            Delayed_Wrappers_Store.Prepend_Enclosing ((Specs_Count   => 0,
                                                       Encl_Specs    => (others => <>),
                                                       Instantiation => Inst,
                                                       T_Info        => The_Info));
         else
            Generate_Wrapper_Body (Wrapper_Name, Generic_Spec, The_Info);
         end if;
      end if;

      --
      -- Nested instantiation: generate instance name as a renaming of duplicated generic in wrapper package
      -- Add a with clause if the instantiation is a library unit!
      --
      Next_Line;
      if Is_Compilation_Unit (Inst) then
         -- Can't use rename, since the target of a library rename must be a library unit
         -- We have generated a wrapper with only the formal parameters, we duplicate the generic here under
         -- the name of the instantiation. A use clause provides direct visibility to the parameters.

         -- Copy the context clauses from the unit that contains the spec of the generic
         declare
            Context_Clauses : constant Element_List := Context_Clause_Elements (Enclosing_Compilation_Unit
                                                                                (Generic_Spec),
                                                                                Include_Pragmas => True);
         begin
            if Context_Clauses /= Nil_Element_List then
               Generic_Span := Element_Span (Context_Clauses (Context_Clauses'First));
               Push_Source (Previous_State, Generic_Span.First_Line, Generic_Span.First_Column);
               Print_Up_To (Context_Clauses (Context_Clauses'Last), Included => True);
               Next_Line;
               Pop_Source (Previous_State);
            end if;
         end;

         -- Duplicate specification skipping the generic formal part
         Generic_Span := Element_Span (Generic_Spec);
         if Is_Nil (Formals) then
            Generic_Span.First_Column := Generic_Span.First_Column + Wide_String'("generic")'Length;
         else
            Generic_Span.First_Line   := Element_Span (Formals (Formals'Last)).Last_Line;
            Generic_Span.First_Column := Element_Span (Formals (Formals'Last)).Last_Column + 1;
         end if;

         Print_Line ("with " & Wrapper_Name & ';');
         Print_Line ("use "  & Wrapper_Name & ';');

         The_Info := (Duplicate    => True,
                      Def_Name     => Simple_Def_Name (Names (Generic_Spec) (1)),
                      Inst_Pos     => Instantiation_Count,
                      Wrapper_Name => To_Unbounded_Wide_String (Wrapper_Name),
                      Copy_Name    => To_Unbounded_Wide_String (Instance_Name));
         Duplicate_Generic_Part (Generic_Spec, The_Info);

         -- Duplicate body
         -- Copy the context clauses from the unit that contains the body of the generic
         if not Is_Nil (Generic_Body) then
            declare
               Context_Clauses : constant Element_List := Context_Clause_Elements (Enclosing_Compilation_Unit
                                                                                   (Generic_Body),
                                                                                   Include_Pragmas => True);
            begin
               if Context_Clauses /= Nil_Element_List then
                  Generic_Span := Element_Span (Context_Clauses (Context_Clauses'First));
                  Push_Source (Previous_State, Generic_Span.First_Line, Generic_Span.First_Column);
                  Print_Up_To (Context_Clauses (Context_Clauses'Last), Included => True);
                  Next_Line;
                  Pop_Source (Previous_State);
               end if;
            end;

            Duplicate_Generic_Part (Generic_Body, The_Info);  -- Same T_Info as the spec
         end if;

      else  -- Not a compilation unit, use rename
         case A_Generic_Instantiation'(Declaration_Kind (Inst)) is
            when A_Procedure_Instantiation =>
               Print ("procedure " & Instance_Name);
               Print_Fixed_For_Formals (Parameter_Profile (Generic_Spec));
            when A_Function_Instantiation =>
               Print ("function " & Instance_Name);
               Print_Fixed_For_Formals (Parameter_Profile (Generic_Spec));
               Print (" return ");
               Print_Fixed_For_Formals (Result_Profile (Generic_Spec));
            when A_Package_Instantiation =>
               Print ("package " & Instance_Name);
         end case;

         Print (" renames " & Wrapper_Name & '.' & Generic_Name & ';');
         Next_Line;
      end if;
      Advance (Inst, Included => True);
      Instantiation_Depth := Instantiation_Depth - 1;
   end Process_Instantiation;

   --------------------------
   -- Process_With_Clauses --
   --------------------------

   procedure Process_With_Clauses (Clauses : Context_Clause_List) is
   -- Eliminate with clauses for generic library packages, except for language defined ones
   -- but add with clauses carried by the generic itself (will be required by the instantiation).
   -- We don't care about generating multiple identical with clauses, since they are harmless.
   -- Nasty case: a generic unit may be a child of a non generic unit. In this case, we must keep the with clause
   -- up to the last non generic unit. This works because fortunately, generic units can have only generic children!
   -- Note: we don't care about use clauses, since there can't be a use clause for a generic unit.
      use Asis.Clauses, Asis.Compilation_Units, Asis.Expressions;
      use Producer, Thick_Queries, Utilities;

      procedure Add_With_Clauses (Unit : Compilation_Unit) is
         use Asis.Text;
         Unit_Clauses : constant Asis.Context_Clause_List := Context_Clause_Elements (Unit);
      begin
         for Clause : Asis.Context_Clause of Unit_Clauses loop
            Print_Line (Trim_All (Element_Image (Clause)));
         end loop;
      end Add_With_Clauses;

   begin  -- Process_With_Clauses
      for Clause : Asis.Context_Clause of Clauses loop
         if Clause_Kind (Clause) = A_With_Clause then
            declare
               Unit_Names : constant Asis.Name_List := Clause_Names (Clause);
               Unit       : Asis.Compilation_Unit;
               Cur_Name   : Asis.Expression;
               Kept_Units : array (Unit_Names'Range) of Asis.Element;
               Keep_With  : Boolean := False;

               -- We assume here that names don't represent more than Max_Generic_Depth generic children...
               Removed_Generics : Asis.Element_List (1 .. Max_Generic_Depth * Unit_Names'Length);
               Removed_Inx      : ASIS_Natural := 0;

               First_Name : Boolean := True;
            begin
               for N in Unit_Names'Range loop
                  Cur_Name := Unit_Names (N);
                  loop
                     Unit := Enclosing_Compilation_Unit (Corresponding_Name_Declaration (Simple_Name (Cur_Name)));
                     exit when Unit_Kind (Unit) not in A_Generic_Unit_Declaration
                       or else Unit_Origin (Unit) /= An_Application_Unit;
                     if Removed_Inx = Max_Generic_Depth then
                        Failure ("Too deep hierarchy of child generics, maximum is "
                                 & ASIS_Integer_Img (Max_Generic_Depth));
                     end if;
                     Removed_Inx := Removed_Inx + 1;
                     Removed_Generics (Removed_Inx) := Simple_Name (Cur_Name);
                     if Expression_Kind (Cur_Name) /= A_Selected_Component then
                        Cur_Name := Nil_Element;
                        exit;
                     end if;
                     Cur_Name := Prefix (Cur_Name);
                  end loop;
                  Kept_Units (N) := Cur_Name;
                  Keep_With      := not Is_Nil (Cur_Name);
               end loop;

               if Keep_With then
                  Print_Up_To (Unit_Names (Unit_Names'First), Included => False);  -- up to the "with", included

                  -- Eliminate names and the following comma, until we find one to be kept
                  -- Then, keep the commas in order to have a proper list
                  for N in Unit_Names'Range loop
                     if not Is_Nil (Kept_Units (N)) then
                        Print_Up_To (Kept_Units (N), Included => True);
                        Advance (Unit_Names (N), Included => True);  -- Skip the rest of the name
                        First_Name := False;
                     elsif First_Name and N /= Unit_Names'Last then
                        Advance (Unit_Names (N + 1), Included => False);  -- eliminate separator comma
                     else
                        Advance (Unit_Names (N), Included => True);
                     end if;
                  end loop;
                  Print_Up_To (Clause, Included => True);  -- print semi-colon
               else
                  -- Eliminate the whole with clause
                  Print_Up_To (Clause, Included => False);
                  Advance (Clause, Included => True);
               end if;

               -- Add with clauses of removed units
               for Removed : Asis.Element of Removed_Generics (1 .. Removed_Inx) loop
                  declare
                     -- Note: since these units are generic, there is always an explicit spec
                     U_Spec : constant Compilation_Unit := Enclosing_Compilation_Unit
                                                            (Corresponding_Name_Declaration (Removed));
                     U_Body : constant Compilation_Unit := Corresponding_Body (U_Spec);
                  begin
                     Add_With_Clauses (U_Spec);
                     if not Is_Nil (U_Body) then
                        Add_With_Clauses (U_Body);
                     end if;
                  end;
               end loop;
            end;
         end if;
      end loop;
   end Process_With_Clauses;

   ----------------------------------------------------------------
   --                 Exported subprograms                       --
   ----------------------------------------------------------------

   -------------
   -- Prepare --
   -------------

   procedure Prepare is
   begin
      Delayed_Wrappers_Store.Activate;
   end Prepare;

   -------------
   -- Process --
   -------------

   procedure Process (Unit_Name           : in     Wide_String;
                      Unit_Rank           : in     Natural;
                      Process_Spec        : in     Boolean;
                      Process_Body        : in     Boolean;
                      Print_Changed_Lines : in     Boolean;
                      Line_Length         : in     Natural;
                      Output_Prefix       : in     String;
                      Keep_Unchanged      : in     Boolean;
                      Overwrite_Option    : in     Boolean)
   is
      use Asis.Compilation_Units;
      use Utilities;
      use Ada.Wide_Text_IO;
      F_Out : File_Type;

      procedure Do_Process (My_Unit : Compilation_Unit) is
         use Ada.Characters.Handling, Producer;
         function Clean_Name (Name : Wide_String) return String is
         begin
            for I in reverse Name'Range loop
               if Name (I) = '/' or Name (I) = '\' then
                  return To_String (Name (I + 1 .. Name'Last));
               end if;
            end loop;
            return To_String (Name);
         end Clean_Name;

         My_Declaration : Declaration;
         The_Control    : Traverse_Control := Continue;
         The_Info       : Traverse_Info;
         Had_Changes    : Boolean;
         Out_Name       : constant String := Output_Prefix & Clean_Name (Text_Name (My_Unit));
      begin   -- Do_Process
         if Output_Prefix /= "" then
            Safe_Open  (F_Out,
                        Name             => Out_Name,
                        Mode             => Create,
                        Overwrite_Option => Overwrite_Option);
            Set_Output (F_Out);
         end if;

         Process_Unit :
         begin
            Scope_Manager.Enter_Unit (My_Unit);

            Process_With_Clauses (Context_Clause_Elements (Compilation_Unit => My_Unit, Include_Pragmas => False));
            Scope_Manager.Exit_Context_Clauses;

            My_Declaration := Unit_Declaration (My_Unit);
            The_Info := (Duplicate    => False,
                         Def_Name     => Nil_Element,
                         Inst_Pos     => 0,
                         Wrapper_Name => Null_Unbounded_Wide_String,
                         Copy_Name    => Null_Unbounded_Wide_String);
            Traverse (My_Declaration, The_Control, The_Info);
            if The_Control /= Terminate_Immediately then
               Print_Up_To (My_Declaration, Included => True, Final => True);
            end if;

            Scope_Manager.Exit_Unit (My_Unit);
         end Process_Unit;

         Producer.Finish (Had_Changes);

         if Is_Open (F_Out) then
            Set_Output (Standard_Output);
            if Had_Changes or Keep_Unchanged then
               Close (F_Out);
            else
               Delete (F_Out);
            end if;
         end if;

      exception
         when Overwrite_Error =>
            User_Message ("File " & To_Wide_String (Out_Name)
                          & " already exists, use -w to overwrite");
         when others =>
            Producer.Finish (Had_Changes);   -- To have all the context

            if Is_Open (F_Out) then
               Set_Output (Standard_Output);
               Close (F_Out);
            end if;
            raise;
      end Do_Process;

      Unit_Decl : Asis.Compilation_Unit := Library_Unit_Declaration (Unit_Name, Asis_Context);
      Has_Spec,
      Has_Body  : Boolean := False;
   begin  -- Process
      if Unit_Kind (Unit_Decl) in A_Generic_Unit_Declaration then
         User_Log ('(' & Integer_Img (Unit_Rank) & '/' & Integer_Img (Units_List.Length) & ')'
                   & " Generic library unit " & Unit_Name & " ignored");
         return;
      end if;

      Producer.Initialize (Line_Length, Print_Changed_Lines);

      if Process_Spec and not Is_Nil (Unit_Decl) then
         User_Log ('(' & Integer_Img (Unit_Rank) & '/' & Integer_Img (Units_List.Length) & ')'
                   & " Instantiating " & Unit_Name & " specification");
         Has_Spec := True;
         Instantiation_Count := 0;
         Do_Process (Unit_Decl);
         User_Log ("   # of instantiations:" & Natural'Wide_Image (Instantiation_Count));
      end if;

      Unit_Decl := Compilation_Unit_Body (Unit_Name, Asis_Context);
      if Process_Body and not Is_Nil (Unit_Decl) then
         Has_Body := True;
         User_Log ('(' & Integer_Img (Unit_Rank) & '/' & Integer_Img (Units_List.Length) & ')'
                   & " Instantiating " & Unit_Name & " body");
         Instantiation_Count := 0;
         Do_Process (Unit_Decl);
         User_Log ("   # of instantiations:" & Natural'Wide_Image (Instantiation_Count));
      end if;

      if not Has_Spec and not Has_Body then
         User_Log ('(' & Integer_Img (Unit_Rank) & '/' & Integer_Img (Units_List.Length) & ')'
                   & Unit_Name & " not found");
      end if;

   exception
      when others =>
         if Is_Open (F_Out) then
            Set_Output (Standard_Output);
            Close (F_Out);
         end if;
         raise;
   end Process;
end Tools.Instantiator;
