----------------------------------------------------------------------
--  Tools.Unuser - Package body                                     --
--  Copyright (C) 2002, 2009 Adalog                                 --
--  Author: J-P. Rosen                                              --
--                                                                  --
--  ADALOG   is   providing   training,   consultancy,   expertise, --
--  assistance and custom developments  in Ada and related software --
--  engineering techniques.  For more info about our services:      --
--  ADALOG                   Tel: +33 1 41 24 31 40                 --
--  19-21 rue du 8 mai 1945  Fax: +33 1 41 24 07 36                 --
--  94110 ARCUEIL            E-m: info@adalog.fr                    --
--  FRANCE                   URL: http://www.adalog.fr              --
--                                                                  --
--  This  unit is  free software;  you can  redistribute  it and/or --
--  modify  it under  terms of  the GNU  General Public  License as --
--  published by the Free Software Foundation; either version 2, or --
--  (at your  option) any later version.  This  unit is distributed --
--  in the hope  that it will be useful,  but WITHOUT ANY WARRANTY; --
--  without even the implied warranty of MERCHANTABILITY or FITNESS --
--  FOR A  PARTICULAR PURPOSE.  See the GNU  General Public License --
--  for more details.   You should have received a  copy of the GNU --
--  General Public License distributed  with this program; see file --
--  COPYING.   If not, write  to the  Free Software  Foundation, 59 --
--  Temple Place - Suite 330, Boston, MA 02111-1307, USA.           --
--                                                                  --
--  As  a special  exception, if  other files  instantiate generics --
--  from  this unit,  or you  link this  unit with  other  files to --
--  produce an executable,  this unit does not by  itself cause the --
--  resulting executable  to be covered  by the GNU  General Public --
--  License.  This exception does  not however invalidate any other --
--  reasons why  the executable  file might be  covered by  the GNU --
--  Public License.                                                 --
----------------------------------------------------------------------

with   -- Standard Ada units
  Ada.Characters.Handling,
  Ada.Wide_Text_IO;

with   -- ASIS components
  Asis.Clauses,
  Asis.Compilation_Units,
  Asis.Declarations,
  Asis.Definitions,
  Asis.Elements,
  Asis.Expressions,
  Asis.Iterator,
  Asis.Statements;
pragma Elaborate_All (Asis.Iterator);

with   -- Adalog reusable components
  Scope_Manager,
  Thick_Queries,
  Utilities;

with   -- Application specific units
  Producer;
package body Tools.Unuser is
   use Asis, Asis.Elements;

   ----------------------------------------------------------------
   --                 Internal elements                          --
   ----------------------------------------------------------------

   -------------------------
   -- Expanded_Name_Image --
   -------------------------

   function Prefix_Image (Name : Asis.Expression) return Wide_String is
   -- Returns the image of the prefix of Name (if Name had been declared as a full name), if Name is
   -- declared in a package, recursively if the enclosing package is itself defined in a package
   --
   --  Appropriate Element_Kinds:
   --       An_Expression => A_Name
   --       A_Defining_Name

      use Asis.Declarations, Asis.Expressions;
      use Thick_Queries, Utilities;

      Decl : Asis.Declaration;
      Encl : Asis.Element;
   begin
      if Element_Kind (Name) = A_Defining_Name then
         Decl := Enclosing_Element (Name);
      else
         Decl := Corresponding_Name_Declaration (Name);
         if Declaration_Kind (Decl) = An_Enumeration_Literal_Specification then
            -- The literal declaration is in an enumeration type definition in the type declaration
            Decl := Enclosing_Element (Enclosing_Element (Decl));
         end if;
      end if;

      if Is_Nil (Decl) then
         -- TBSL case where Decl is a dispatching call

         -- Predefined operator => Get the declaration from the parameters' type
         Encl := Enclosing_Element (Name);
         while Expression_Kind (Encl) /= A_Function_Call loop
            Encl := Enclosing_Element (Encl);
         end loop;
         Decl := Corresponding_Expression_Type (Actual_Parameter (Function_Call_Parameters (Encl) (1)));
      end if;

      Encl := Enclosing_Element (Decl);
      if Is_Nil (Encl) then
         -- Name was a compilation unit
         -- There is a prefix only if it was a child unit
         declare
            Unit_Name : constant Asis.Defining_Name := Names (Decl) (1);
         begin
            if Defining_Name_Kind (Unit_Name) = A_Defining_Expanded_Name then
               return Prefix_Image (Simple_Name (Defining_Prefix (Unit_Name)))
                    & Name_Image (Simple_Name (Defining_Prefix (Unit_Name))) & '.';
            else
               return "";
            end if;
         end;
      end if;

      if Declaration_Kind (Encl) /= A_Package_Declaration then
         return "";
      end if;

      declare
         Pack_Name : constant Asis.Defining_Name := Names (Encl) (1);
         Encl_Name : constant Wide_String := (if Defining_Name_Kind (Pack_Name) = A_Defining_Expanded_Name
                                              then Defining_Name_Image (Pack_Name)
                                              else Prefix_Image (Pack_Name) & Defining_Name_Image (Pack_Name));
         -- NB: Defining_Name_Image of a Defining_Expanded_Name includes the prefixes, and can happen only for
         -- library units
      begin
         if To_Upper (Encl_Name) = "STANDARD"
           or else Scope_Manager.Is_Active (Encl)
         then
            -- Directly visible
            return "";
         else
            return Encl_Name & '.';
         end if;
      end;
   end Prefix_Image;

   -----------------
   -- In_Scope_Of --
   -----------------

   subtype Use_Type_Kinds is Clause_Kinds range A_Use_Type_Clause .. A_Use_All_Type_Clause;
   type Use_Info is
      record
         Used_Decl : Asis.Declaration;
         Kind      : Use_Type_Kinds;
      end record;

   function Equivalent_Keys (L, R : Use_Info) return Boolean is (Is_Equal (L.Used_Decl, R.Used_Decl));
   package Use_Scoping is new Scope_Manager.Scoped_Store (Use_Info, Equivalent_Keys);

   function Is_Use_Type_Visible (Name : Asis.Name; Use_Kind : Use_Type_Kinds) return Boolean is
   -- Tells if Name is visible thanks to a use [all] type clause, or wider, i.e:
   -- Use_Kind = Use_Type_Clause, considers also Use_All_Type_Clause
      use Thick_Queries;
      use Asis.Declarations, Asis.Expressions;

      function Has_Use_Type (The_Type : Asis.Definition) return Boolean is
         use Scope_Manager;
      begin
         Use_Scoping.Reset ((The_Type, Use_Kind), All_Scopes);
         while Use_Scoping.Data_Available loop
            case Use_Kind is
               when A_Use_Type_Clause =>
                  return True;
               when A_Use_All_Type_Clause =>
                  if  Use_Scoping.Current_Data.Kind = A_Use_All_Type_Clause then
                     return True;
                  end if;
            end case;
            Use_Scoping.Next;
         end loop;
         return False;
      end Has_Use_Type;

   begin  -- Is_Use_Type_Visible
      if Is_Nil (Corresponding_Name_Declaration (Name)) then
         -- some predefined stuff, without a declaration. It's either declared in Standard (and we don't care), or
         -- it is an operator of a user defined numeric type. The latter is a tricky case: we'll assume that it is
         -- a primitive operation of its first argument (not the result type, because of fixed points!)
         if Expression_Kind (Name) /= An_Operator_Symbol then
            return False;
         end if;

         declare
            Call : constant Asis.Element := Enclosing_Element (Name);
         begin
            case Expression_Kind (Call) is
               when A_Selected_Component =>
                  -- Cannot be use-visible
                  return False;
               when A_Function_Call =>
                  return Has_Use_Type (First_Defining_Name
                                       (Names
                                          (Corresponding_Expression_Type
                                             (Actual_Parameter
                                                (Function_Call_Parameters (Call) (1))
                                               )
                                            ) (1)));
               when others =>
                  -- Could be an actual in a generic instantiation... give up, this creates only spurious named
                  -- notations (harmless)
                  return False;
            end case;
         end;

      elsif Callable_Kind (Name) in A_Procedure_Callable ..  A_Function_Callable then
         declare
            Profile : constant Profile_Descriptor := Types_Profile (Corresponding_Name_Declaration (Name));
         begin
            -- if one of the parameters' (or result) type is the target of a use all type, we assume the
            -- subprogram is primitive for it, and therefore no expansion of the SP name is needed.
            -- TBSL: counter examples?
            for P : Profile_Entry of Profile.Formals loop
               if Has_Use_Type (First_Defining_Name (P.Name)) then
                  return True;
               end if;
            end loop;
            if not Is_Nil (Profile.Result_Type.Name) then
               if Has_Use_Type (First_Defining_Name (Profile.Result_Type.Name)) then
                  return True;
               end if;
            end if;
            return False;
         end;

      else
         return Has_Use_Type (First_Defining_Name (Names (Corresponding_First_Subtype
                                                          (Corresponding_Expression_Type (Name))) (1)));
      end if;

   end Is_Use_Type_Visible;


   --------------
   -- Traverse --
   --------------
   type Info is null record;

   procedure Pre_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Info);
   procedure Post_Procedure
     (Element : in     Asis.Element;
      Control : in out Asis.Traverse_Control;
      State   : in out Info);
   procedure Traverse is new Asis.Iterator.Traverse_Element
     (Info, Pre_Procedure, Post_Procedure);

   -------------------
   -- Pre_Procedure --
   -------------------

   procedure Pre_Procedure (Element : in     Asis.Element;
                            Control : in out Asis.Traverse_Control;
                            State   : in out Info)
   is
      use Thick_Queries;
      use Producer, Scope_Manager;
      use Asis.Clauses, Asis.Declarations, Asis.Definitions, Asis.Expressions;

      -- This is used to traverse manually the children of an element that includes a private part,
      -- in order to separate the private part from the visible one and call Enter_Private_Part in between.
      -- Note that Enter_Private_Part is called even if there is no private part.
      -- This replaces then normal (recursive) traversal, any code that calls this procedure must
      -- set Control to Abandon_Children
      procedure Traverse_With_Private (Visible_Part : in     Asis.Declarative_Item_List;
                                       Private_Part : in     Asis.Declarative_Item_List;
                                       Priv_Control : in out Asis.Traverse_Control;
                                       Priv_State   : in out Info)
      is
         use Utilities;
      begin

         for Item : Asis.Declaration of Visible_Part loop
            Traverse (Item, Priv_Control, Priv_State);
            case Priv_Control is
               when Continue =>
                  null;
               when Terminate_Immediately =>
                  return;
               when Abandon_Children =>
                  Failure ("Traverse returned Abandon_Children-1");
               when Abandon_Siblings =>
                  Priv_Control := Continue;
                  return;
            end case;
         end loop;

         Scope_Manager.Enter_Private_Part;

         for Item : Asis.Declaration of Private_Part loop
            Traverse (Item, Priv_Control, Priv_State);
            case Priv_Control is
               when Continue =>
                  null;
               when Terminate_Immediately =>
                  return;
               when Abandon_Children =>
                  Failure ("Traverse returned Abandon_Children-2");
               when Abandon_Siblings =>
                  Priv_Control := Continue;
                  return;
            end case;
         end loop;
      end Traverse_With_Private;

      Mark  : Asis.Element;
      Param : Asis.Expression;
   begin     -- Pre_Procedure
      case Element_Kind (Element) is
         when A_Clause =>
            case Clause_Kind (Element) is
               when A_Use_Package_Clause =>
                  Print_Up_To (Element, Included => False);
                  Advance (Element);
                  -- We don't want to traverse the names of packages (formerly) used
                  Control := Abandon_Children;
               when A_Use_Type_Clause | A_Use_All_Type_Clause =>
                  if Use_Type_Option then
                     for Name : Asis.Name of Clause_Names (Element) loop
                        Use_Scoping.Push ((Used_Decl => First_Defining_Name (Name), Kind => Clause_Kind (Element)));
                     end loop;
                  else
                     Print_Up_To (Element, Included => False);
                     Advance (Element);
                     -- We don't want to traverse the names of packages (formerly) used
                     Control := Abandon_Children;
                  end if;
               when others =>
                  null;
            end case;

         when A_Declaration =>
            case Declaration_Kind (Element) is
               when A_Function_Declaration
                  | An_Expression_Function_Declaration   -- Ada 2012
                  | A_Procedure_Declaration
                  | A_Null_Procedure_Declaration
                  | An_Entry_Declaration
                  | A_Generic_Procedure_Declaration
                  | A_Generic_Function_Declaration
                  | A_Formal_Procedure_Declaration
                  | A_Formal_Function_Declaration
                  | A_Package_Body_Declaration
                  | A_Task_Type_Declaration
                  | A_Single_Task_Declaration
                  | A_Protected_Type_Declaration
                  | A_Single_Protected_Declaration
                  | A_Task_Body_Declaration
                  | A_Protected_Body_Declaration
                  | An_Entry_Body_Declaration
                  | A_Procedure_Body_Declaration
                  | A_Function_Body_Declaration
                  | A_Package_Renaming_Declaration
                  | A_Procedure_Renaming_Declaration
                  | A_Function_Renaming_Declaration
                  | A_Generic_Package_Renaming_Declaration
                  | A_Generic_Procedure_Renaming_Declaration
                  | A_Generic_Function_Renaming_Declaration
                  | A_Package_Instantiation
                  | A_Procedure_Instantiation
                  | A_Function_Instantiation
                  =>
                  Scope_Manager.Enter_Scope (Element);

               when A_Package_Declaration => -- Thing that can have a private part
                  Enter_Scope (Element);
                  Traverse_With_Private (Names (Element) (1)
                                         & Visible_Part_Declarative_Items (Element, Include_Pragmas => True),
                                         Private_Part_Declarative_Items (Element, Include_Pragmas => True),
                                         Control,
                                         State);

                  -- Post-procedure is not automatically called when exiting
                  -- with Control = Abandon_Children:
                  Post_Procedure (Element, Control, State);
                  Control := Abandon_Children;

               when A_Generic_Package_Declaration => -- Thing that can have a private part
                  Enter_Scope (Element);
                  Traverse_With_Private (Generic_Formal_Part (Element, Include_Pragmas => True)
                                         & Names(Element)(1)
                                         & Visible_Part_Declarative_Items (Element, Include_Pragmas => True),
                                         Private_Part_Declarative_Items (Element, Include_Pragmas => True),
                                         Control,
                                         State);

                  -- Post-procedure is not automatically called when exiting
                  -- with Control = Abandon_Children:
                  Post_Procedure (Element, Control, State);
                  Control := Abandon_Children;

               when others =>
                  null;
            end case;

         when A_Definition =>
            case Definition_Kind (Element) is
               when An_Aspect_Specification =>
                  -- Do not traverse the aspect mark, unless it is an attribute whose prefix (only!) should
                  -- be traversed
                  Mark := Aspect_Mark (Element);
                  if Expression_Kind (Mark) = An_Attribute_Reference then
                     Traverse (Prefix (Mark), Control, State);
                  end if;
                  if Control not in Terminate_Immediately | Abandon_Children then
                     Traverse (Aspect_Definition (Element), Control, State);
                  end if;
                  if Control = Terminate_Immediately then
                     return;
                  end if;
                  Control := Abandon_Children;

               when A_Task_Definition   -- Things that can have a private part
                 | A_Protected_Definition
                 =>
                  Traverse_With_Private (Visible_Part_Items (Element, Include_Pragmas => True),
                                         Private_Part_Items (Element, Include_Pragmas => True),
                                         Control,
                                         State);

                  -- Post-procedure is not automatically called when exiting
                  -- with Control = Abandon_Children:
                  Post_Procedure (Element, Control, State);
                  Control := Abandon_Children;

               when others =>
                  null;
            end case;

         when An_Expression =>
            case Expression_Kind (Element) is
               when An_Identifier
                  | A_Character_Literal
                  =>
                     -- These are not the Rhs of a selected component, because they are handled below
                  Print_Up_To (Element, Included => False);
                  Print (Prefix_Image (Element));

               when An_Explicit_Dereference =>
                  Traverse (Prefix (Element), Control, State);
                  if Control = Terminate_Immediately then
                     return;
                  end if;
                  Control := Abandon_Children;

               when An_Enumeration_Literal =>
                  if not Use_Type_Option or else not Is_Use_Type_Visible (Element, A_Use_All_Type_Clause) then
                     Print_Up_To (Element, Included => False);
                     Print (Prefix_Image (Element));
                  end if;

               when An_Operator_Symbol =>
                  if not Use_Type_Option or else not Is_Use_Type_Visible (Element, A_Use_Type_Clause) then
                     Print_Up_To (Element, Included => False);
                     Print (Prefix_Image (Element));
                  end if;

               when A_Selected_Component =>
                  Traverse (Prefix (Element), Control, State);
                  if Control = Terminate_Immediately then
                     return;
                  end if;
                  declare
                     Rhs : constant Asis.Expression := Selector (Element);
                  begin
                     case Expression_Kind (Rhs) is
                        when An_Identifier
                           | An_Operator_Symbol
                           | A_Character_Literal
                           | An_Enumeration_Literal
                           | An_Explicit_Dereference
                           =>
                           -- Do not expand these Rhs!
                           Print_Up_To (Rhs, Included => True);

                        when A_Function_Call =>
                           -- Cannot be in infix notation
                           Print_Up_To (Prefix (Rhs), Included => True);
                           declare
                              Params : constant Association_List := Function_Call_Parameters (Rhs);
                           begin
                              for Assoc : Association of Params loop
                                 Traverse (Assoc, Control, State);
                                 if Control = Terminate_Immediately then
                                    return;
                                 end if;
                              end loop;
                           end;
                        when An_Indexed_Component =>
                           Print_Up_To (Rhs, Included => True);
                           declare
                              Indexes : constant Expression_List := Index_Expressions (Rhs);
                           begin
                              for Index : Asis.Expression of Indexes loop
                                 Traverse (Index, Control, State);
                                 if Control = Terminate_Immediately then
                                    return;
                                 end if;
                              end loop;
                           end;
                        when A_Slice =>
                           Traverse (Slice_Range (Rhs), Control, State);
                           if Control = Terminate_Immediately then
                              return;
                           end if;
                        when others =>
                           Utilities.Failure ("Unexpected selector", Rhs);
                     end case;
                     Control := Abandon_Children;
                  end;

               when A_Function_Call =>
                  -- Only infix calls are treated here, we let recurse into the function name
                  -- for others.
                  if not Is_Prefix_Call (Element)
                    and then (not Use_Type_Option
                              or else not Is_Use_Type_Visible (Simple_Name (Prefix (Element)), A_Use_Type_Clause))
                  then
                     -- Turn it into an expanded prefix call (must be an operator)
                     declare
                        Op     : constant Asis.Expression := Prefix (Element);
                        Op_Pfx : constant Wide_String     := Prefix_Image (Op);
                     begin
                        if Op_Pfx /= "" then
                           -- Keep predefined operators in infix notation
                           case Operator_Kind (Op) is
                              when A_Unary_Plus_Operator
                                 | A_Unary_Minus_Operator
                                 | An_Abs_Operator
                                 | A_Not_Operator
                                 =>
                                 Print_Up_To (Op, Included => False);
                                 Print (Op_Pfx);
                                 Print ("""");
                                 Print_Up_To (Op, Included => True);
                                 Print (""" (");
                                 Param := Function_Call_Parameters (Element) (1);
                                 Traverse (Param, Control, State);
                                 Print_Up_To (Param, Included => True);
                              when others =>
                                 Print_Up_To (Element, Included => False);
                                 Print (Op_Pfx);
                                 Print (Name_Image (Op));
                                 Print (" (");

                                 Param := Function_Call_Parameters (Element) (1);
                                 Traverse (Param, Control, State);
                                 Print_Up_To (Param, Included => True);
                                 Advance (Op, Included => True);
                                 Print (",");
                                 Param := Function_Call_Parameters (Element) (2);
                                 Traverse (Param, Control, State);
                                 Print_Up_To (Param, Included => True);
                           end case;
                           Print (")");

                           Control := Abandon_Children;
                        end if;
                     end;
                  elsif Is_Prefix_Notation (Element) then
                     -- There is necessarily at least one parameter
                     -- Traverse first parameter, then function name, then other parameters
                     declare
                        Params : constant Association_List := Function_Call_Parameters (Element);
                     begin
                        Traverse (Params (1), Control, State);
                        if Control = Terminate_Immediately then
                           return;
                        end if;
                        Print_Up_To (Prefix (Element), Included => True);
                        for Assoc : Asis.Association of Params (2 .. Params'Last) loop
                           Traverse (Assoc, Control, State);
                           if Control = Terminate_Immediately then
                              return;
                           end if;
                        end loop;
                     end;
                     Control := Abandon_Children;
                  end if;

               when An_Attribute_Reference =>
                  -- Don't traverse the attribute itself
                  Traverse (Prefix (Element), Control, State);
                  Control := Abandon_Children;

               when others =>
                  null;
            end case;  -- Expression_Kind

         when A_Statement =>
            case Statement_Kind (Element) is
               when A_For_Loop_Statement
                  | A_Block_Statement
                  | An_Accept_Statement
                  | An_Extended_Return_Statement
                  =>
                  Enter_Scope (Element);

               when A_Procedure_Call_Statement | An_Entry_Call_Statement =>
                  if Is_Prefix_Notation (Element) then
                     -- There is necessarily at least one parameter
                     -- Traverse first parameter, then callable name, then other parameters
                     declare
                        use Asis.Statements;
                        Params : constant Association_List := Call_Statement_Parameters (Element);
                     begin
                        Traverse (Params (1), Control, State);
                        if Control = Terminate_Immediately then
                           return;
                        end if;
                        Print_Up_To (Called_Name (Element), Included => True);
                        for Assoc : Asis.Association of Params (2 .. Params'Last) loop
                           Traverse (Assoc, Control, State);
                           if Control = Terminate_Immediately then
                              return;
                           end if;
                        end loop;
                     end;
                     Control := Abandon_Children;
                  end if;
               when others =>
                  null;
            end case;

         when An_Exception_Handler =>
            Enter_Scope (Element);

         when A_Pragma =>
            -- Don't traverse pragmas at all
            Control := Abandon_Children;

         when others =>
            null;
      end case;  -- Element_Kind
   end Pre_Procedure;

   --------------------
   -- Post_Procedure --
   --------------------

   procedure Post_Procedure (Element : in     Asis.Element;
                             Control : in out Asis.Traverse_Control;
                             State   : in out Info)
   is
      pragma Unreferenced (Control, State);
      use Scope_Manager;

   begin
      case Element_Kind (Element) is
         when A_Declaration =>
            case Declaration_Kind (Element) is
               when A_Function_Declaration
                  | An_Expression_Function_Declaration   -- Ada 2012
                  | A_Procedure_Declaration
                  | A_Null_Procedure_Declaration
                  | An_Entry_Declaration
                  | A_Package_Declaration
                  | A_Generic_Procedure_Declaration
                  | A_Generic_Function_Declaration
                  | A_Generic_Package_Declaration
                  | A_Formal_Procedure_Declaration
                  | A_Formal_Function_Declaration
                  | A_Package_Body_Declaration
                  | A_Task_Type_Declaration
                  | A_Single_Task_Declaration
                  | A_Protected_Type_Declaration
                  | A_Single_Protected_Declaration
                  | A_Task_Body_Declaration
                  | A_Protected_Body_Declaration
                  | An_Entry_Body_Declaration
                  | A_Procedure_Body_Declaration
                  | A_Function_Body_Declaration
                  | A_Package_Renaming_Declaration
                  | A_Procedure_Renaming_Declaration
                  | A_Function_Renaming_Declaration
                  | A_Generic_Package_Renaming_Declaration
                  | A_Generic_Procedure_Renaming_Declaration
                  | A_Generic_Function_Renaming_Declaration
                  | A_Package_Instantiation
                  | A_Procedure_Instantiation
                  | A_Function_Instantiation
                  =>
                  Exit_Scope (Element);
               when others =>
                  null;
            end case;

         when A_Statement =>
            case Statement_Kind (Element) is
               when A_For_Loop_Statement
                  | A_Block_Statement
                  | An_Accept_Statement
                  | An_Extended_Return_Statement
                  =>
                  Exit_Scope (Element);
               when others =>
                  null;
            end case;

         when An_Exception_Handler =>
            Exit_Scope (Element);

         when others =>
            null;
      end case;
   end Post_Procedure;

   ----------------------------------------------------------------
   --                 Exported subprograms                       --
   ----------------------------------------------------------------

   -------------
   -- Process --
   -------------

   procedure Process (Unit_Name           : in     Wide_String;
                      Process_Spec        : in     Boolean;
                      Process_Body        : in     Boolean;
                      Print_Changed_Lines : in     Boolean;
                      Line_Length         : in     Natural;
                      Output_Prefix       : in     String;
                      Keep_Unchanged      : in     Boolean;
                      Overwrite_Option    : in     Boolean)
   is
      use Asis.Compilation_Units;
      use Utilities;
      use Ada.Wide_Text_IO;
      F_Out : File_Type;

      procedure Do_Process (My_Unit : Compilation_Unit) is
         use Ada.Characters.Handling, Producer;
         function Clean_Name (Name : Wide_String) return String is
         begin
            for I in reverse Name'Range loop
               if Name (I) = '/' or Name (I) = '\' then
                  return To_String (Name (I+1 .. Name'Last));
               end if;
            end loop;
            return To_String (Name);
         end Clean_Name;

         My_Declaration : Declaration;
         The_Control    : Traverse_Control := Continue;
         The_Info       : Info := (null record);
         Had_Changes    : Boolean;
         Out_Name       : constant String := Output_Prefix & Clean_Name (Text_Name (My_Unit));
      begin   -- Do_Process
         if Is_Nil (My_Unit) then
            return;
         end if;

         if Output_Prefix /= "" then
            Safe_Open  (F_Out,
                        Name => Out_Name,
                        Mode => Create,
                        Overwrite_Option => Overwrite_Option);
            Set_Output (F_Out);
         end if;

         Scope_Manager.Enter_Unit (My_Unit) ;

      Process_Context_Clauses :
         declare
            My_CC_List  : constant Context_Clause_List
              := Context_Clause_Elements (Compilation_Unit => My_Unit,
                                          Include_Pragmas  => True) ;
         begin
            for Clause : Asis.Context_Clause of My_CC_List loop
               Traverse (Clause, The_Control, The_Info);
            end loop;
         end Process_Context_Clauses;
         Scope_Manager.Exit_Context_Clauses;

      Process_Unit :
         begin
            My_Declaration := Unit_Declaration (My_Unit);
            Traverse (My_Declaration, The_Control, The_Info);
            if The_Control /= Terminate_Immediately then
               Print_Up_To (My_Declaration, Included => True, Final => True);
            end if;
         end Process_Unit;

         Producer.Finish (Had_Changes);
         Scope_Manager.Exit_Unit (My_Unit);

         if Is_Open (F_Out) then
            Set_Output (Standard_Output);
            if Had_Changes or Keep_Unchanged then
               Close (F_Out);
            else
               Delete (F_Out);
            end if;
         end if;

      exception
         when Overwrite_Error =>
            User_Message ("File " & To_Wide_String (Out_Name)
                          & " already exists, use -w to overwrite");
         when others =>
            if Is_Open (F_Out) then
               Set_Output (Standard_Output);
               Close (F_Out);
            end if;
            raise;
      end Do_Process;

   begin  -- Process
      Producer.Initialize (Line_Length, Print_Changed_Lines);
      Use_Scoping.Activate;

      if Process_Spec then
         User_Log ("Unusing " & Unit_Name & " specification");
         Do_Process (Library_Unit_Declaration (Unit_Name, Asis_Context));
      end if;
      if Process_Body then
         User_Log ("Unusing " & Unit_Name & " body");
         Do_Process (Compilation_Unit_Body (Unit_Name, Asis_Context));
      end if;

   exception
      when others =>
         if Is_Open (F_Out) then
            Set_Output (Standard_Output);
            Close (F_Out);
         end if;
         raise;
   end Process;

end Tools.Unuser;
