#!/bin/bash
# Usage:
# ./run.sh [<adasubst options>]

#
# functions
#
function put () {
    local msg="$1"
    local pad=$2
    local ali=$3

    if [ "$3" = "center" ]; then
	lef=$[(${pad}-${#msg})/2]
	printf "%${lef}s%s" "" "${msg}"
	rig=$[${pad}-${#msg}-${lef}]
	printf "%${rig}s" ""
    elif [ "$3" = "right" ]; then
	printf "%-${pad}s" "${msg}"
    else
	printf "%${pad}s" "${msg}"
    fi
}

function put_line () {
    put "$1" $2 $3
    printf "\n"
}

function put_title_line () {
    put "--"
    put "$1" 75 center
    put_line "--"
}

function put_line_line () {
    put      "----------------------------------------"
    put_line "---------------------------------------"
}

function print_time () {
    t=$2
    if [ $t -ge 3600 ]; then
	h=`expr $t / 3600`"h "
	t=`expr $t % 3600`
    else
	h=""
    fi
    if [ $t -ge 60 ]; then
	m=`expr $t / 60`"mn "
	t=`expr $t % 60`
    else
	m=""
    fi
    s=$t"s."
    put_line  "$1 ${h}${m}${s}"
}

########################################################
# Actual beginning
########################################################

#
# Initialization
#
case $(uname) in
    *win*|*Win*|*WIN*)  # Windows, Cygwin...
	EXT=".exe"
	;;
    *)                  # Assume *nix
	EXT=""
	;;
esac

for E in "../src/adasubst$EXT" "../adasubst$EXT" $(which adasubst); do
   if [ -e $E ] ; then
       ADASUBST=$E
       break
   fi
done

if [ -d res ]; then
    rm -rf res/*
    mkdir res/comp
else
    mkdir res
    mkdir res/comp
fi

put_line_line
put_title_line
put_title_line "`${ADASUBST} -h 2>&1  | head -n 1 | tr -d \\\\r`"
put_title_line
put_title_line VALIDATION
put_title_line
put_title_line "$(date)"
put_title_line
put_line_line

run_start=`date +%s`
nb_tests=0
rm -f res/comp/*

#
# Actual tests.
#

# Framework
echo "============= Framework"
test_case=t_folding
${ADASUBST} instantiate ${test_case} -v  \
	| tr -d \\r >>res/${test_case}.txt
nb_tests=$((nb_tests+1))
echo " "


# Help
echo "============= Help"
test_case=t_help
${ADASUBST} help 2>res/${test_case}.txt
sed -i "/with ASIS/d" res/${test_case}.txt
nb_tests=$((nb_tests+1))
echo " "

# Prepare
echo "============= Prepare"
test_case=t_prepare
for f in *.ads ; do
    ${ADASUBST} Prepare ${f%.*} -vo res/${test_case}.txt
done
chmod a+rw res/${test_case}.txt
nb_tests=$((nb_tests+1))
echo " "

# Instantiate
echo "============= Instantiate"
test_case=t_instantiate
rm -f res/${test_case}.txt
for f in ${test_case}*.adb t_instantiate_lib.ads ; do
    echo "------------------ $f ---------------" >> res/${test_case}.txt
    ${ADASUBST} instantiate ${f%.*} -vr \
	| tr -d \\r >>res/${test_case}.txt
    nb_tests=$((nb_tests+1))
done
gnatchop -w res/${test_case}.txt res/comp/
echo " "

# Translate
echo "============= Translate"
test_case=t_translate
rm -f res/${test_case}.txt
for f in ${test_case}_*.adb ; do
    ${ADASUBST} TRANSLATE t_translate.dic ${f%.*} -v \
	| tr -d \\r >>res/${test_case}.txt
    nb_tests=$((nb_tests+1))
done
gnatchop res/${test_case}.txt res/comp/
echo " "

# Unrepresent
echo "============= Unrepresent"
test_case=t_unrep
${ADASUBST} UnRepresent -rvw  $test_case \
	| tr -d \\r >res/${test_case}.txt
gnatchop res/${test_case}.txt res/comp/
${ADASUBST} unrepresent -c -rvw  $test_case \
	| tr -d \\r >>res/${test_case}.txt
nb_tests=$((nb_tests+2))
echo " "

# Unuse
echo "============= Unuse"
test_case=t_unuse
${ADASUBST} UNUSE -rvw  $test_case \
	| tr -d \\r >res/${test_case}.txt
gnatchop res/${test_case}.txt res/comp/
${ADASUBST} unuse -uc -rvw  $test_case \
	| tr -d \\r >>res/${test_case}.txt
nb_tests=$((nb_tests+2))
echo " "


#
# Tests finalization
#

run_stop=`date +%s`
pushd ref 1>/dev/null
list=`ls *.txt`
popd 1>/dev/null

dos2unix -q res/*

nb_passed=0
nb_failed=0
put_line_line
put_title_line "Test result"
put_line_line
for test_case in $list; do
    diff=`diff --strip-trailing-cr res/${test_case} ref/${test_case} 2>&1 `
    if [ "$diff" == "" ]; then
	nb_passed=$((nb_passed+1))
	printf "=> %-60s%-13s <=\n" ${test_case} "       PASSED"
    else
	nb_failed=$((nb_failed+1))
	printf "=> %-60s%-13s <=\n" ${test_case} "FAILED       "
    fi;
done

put_line_line
put_title_line "Passed= $nb_passed, Failed= $nb_failed"

# Check compilation
rm -f res/comp/*_fail*
gcc -c -gnatwA res/comp/*.adb 2>res/comp/compilation.log
if [ $? -eq 0 ] ; then
    put_title_line "Compilation check succesful"
else
    put_title_line "Generated code did not compile !"
    put_title_line "Log file: res/comp/compilation.log"
fi
put_title_line "Checked executable: $ADASUBST ($(stat --printf=%y $ADASUBST|cut -d '.' -f 1))"
put_title_line "`${ADASUBST} -h 2>&1  | head -n 1 | tr -d \\\\r`"
  
put_line_line
print_time "Total run time: " `expr $run_stop - $run_start`

rm -f *.o *.ali *.adt

if [ $nb_failed -ne 0 ] ; then
    . failed.sh
fi

